﻿using Hims.Api.Models;
using Hims.Api.Utilities;
using Hims.Domain.Helpers;
using Hims.Domain.Services;
using Hims.Shared.DataFilters;
using Hims.Shared.EntityModels;
using Hims.Shared.Library.Enums;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{

    /// <inheritdoc />
    [Route("api/diet-items")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class DietItemsController : BaseController
    {
        /// <summary>
        /// The auditlog services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <summary>
        /// The auditlog services.
        /// </summary>
        private readonly IAESHelper aeshelper;


        private readonly IDietItemsService dietItemsService;

        public DietItemsController(
           IAuditLogService auditLogServices,
           IDietItemsService dietItemsServices,
           IAESHelper aeshelper)
        {

            this.auditLogServices = auditLogServices;
            this.dietItemsService = dietItemsServices;
            this.aeshelper = aeshelper;
        }

        [HttpPost]
        [Route("fetch-items")]
        public async Task<ActionResult> FetchItems([FromBody] DietItemsModel model)
        {
            var response = await this.dietItemsService.FetchItemsAsync(model);
            return this.Success(response);
        }
        [HttpPost]
        [Route("fetch-measure")]
        public async Task<ActionResult> FetchMeasure(DietItemsModel model)
        {
            var response = await this.dietItemsService.FetchMeasureAsync(model);
            return this.Success(response);

        }

        [HttpPost]
        [Route("add")]
        public async Task<ActionResult> InsertAsync([FromBody] DietItemsModel model, [FromHeader] LocationHeader header)
        {
            model = (DietItemsModel)EmptyFilter.Handler(model);
            var response = await this.dietItemsService.InsertItemsAsync(model).ConfigureAwait(false);

            if (response > 0)
            {
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.DietItems,
                    LogFrom = (short)model.LoginRoleId,
                    LocationId = Convert.ToInt32(header.LocationId),
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $"<b>{model.CreatedByName}</b> has added <b>diet item Name</b> of <strong>{model.DietItemsName}</strong> successfully."
                };
                await this.auditLogServices.LogAsync(auditLogModel);

            }

            return this.Success(response);

        }
        [HttpPut]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] DietItemsModel model , [FromHeader] LocationHeader header)
        {
            model = (DietItemsModel)EmptyFilter.Handler(model);

            var response = await this.dietItemsService.UpdateAsync(model);
            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.DietItems,
                    LogDate = DateTime.Now,
                    LocationId = Convert.ToInt32(header.LocationId),
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LogDescription = $"<b>{model.CreatedByName}</b> has updated <b>Diet Item Name</b> of <strong>{model.DietItemsName}</strong> successfully."

                };
                await this.auditLogServices.LogAsync(auditLog);
            }

            return this.Success(response);
        }

        [HttpPost]
        [Route("modify-status")]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] DietItemsModel model, [FromHeader] LocationHeader header)
        {
            model = (DietItemsModel)EmptyFilter.Handler(model);
            var response = await this.dietItemsService.ModifyStatusAsync(model);
            if (response > 0)
            {
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.DietItems,
                    LogDate = DateTime.Now,
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LocationId = Convert.ToInt32(header.LocationId),
                    LogDescription = model.Active == true ? $@"<b>{model.ModifiedByName}</b> has Activated status of diet item <br/>name:<b> '{model.DietItemsName}'</b> on {DateTime.UtcNow.AddMinutes(330)}" : $@"<b>{model.ModifiedByName}</b> has Deactivated status of diet item <br/>name: <b>'{model.DietItemsName}'</b> on {DateTime.UtcNow.AddMinutes(330)}",

                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }
            return this.Success(response);

        }

    }
}
